import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Music, Star, Calendar, User, TrendingUp, Zap, Heart } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyClasses.css'

const MyClasses = () => {
  const [classes, setClasses] = useState([])

  useEffect(() => {
    loadClasses()
    const interval = setInterval(loadClasses, 500)
    return () => clearInterval(interval)
  }, [])

  const loadClasses = () => {
    const saved = localStorage.getItem('danceClasses')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setClasses(parsed)
      } catch (e) {
        setClasses([])
      }
    } else {
      const defaultClasses = [
        {
          id: 1,
          style: 'Хип-хоп',
          instructor: 'Алексей Воронов',
          date: '2025-03-28',
          rating: 5
        },
        {
          id: 2,
          style: 'Бачата',
          instructor: 'Мария Соколова',
          date: '2025-03-26',
          rating: 5
        },
        {
          id: 3,
          style: 'Зумба',
          instructor: 'Елена Петрова',
          date: '2025-03-24',
          rating: 4
        }
      ]
      setClasses(defaultClasses)
      localStorage.setItem('danceClasses', JSON.stringify(defaultClasses))
    }
  }

  const totalClasses = classes.length
  const uniqueInstructors = new Set(classes.map(c => c.instructor)).size
  const averageRating = classes.length > 0 
    ? (classes.reduce((sum, c) => sum + c.rating, 0) / classes.length).toFixed(1)
    : 0

  return (
    <div className="my-classes">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7 }}
        >
          <motion.div 
            className="hero-icon-wrapper"
            initial={{ scale: 0, rotate: -360 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.4 }}
          >
            <Music size={120} strokeWidth={2.5} className="hero-icon" />
          </motion.div>
          <h1 className="hero-title">Мои занятия</h1>
          <p className="hero-description">
            Ваш прогресс в танцевальных занятиях и достижения
          </p>
        </motion.section>

        <motion.div 
          className="stats-grid"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.3 }}
        >
          <motion.div 
            className="stat-card"
            whileHover={{ y: -8, rotate: 2, transition: { duration: 0.3 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(230, 57, 70, 0.25)' }}>
              <Music size={56} strokeWidth={2.5} color="#E63946" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{totalClasses}</div>
              <div className="stat-label">Занятий</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ y: -8, rotate: -2, transition: { duration: 0.3 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(247, 127, 0, 0.25)' }}>
              <Star size={56} strokeWidth={2.5} color="#F77F00" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{averageRating}</div>
              <div className="stat-label">Средняя оценка</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ y: -8, rotate: 2, transition: { duration: 0.3 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(252, 191, 73, 0.25)' }}>
              <User size={56} strokeWidth={2.5} color="#FCBF49" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{uniqueInstructors}</div>
              <div className="stat-label">Преподавателей</div>
            </div>
          </motion.div>
        </motion.div>

        <ImageSlider images={[
          '/images/dance-1.jpg',
          '/images/dance-2.jpg',
          '/images/dance-3.jpg',
          '/images/dance-4.jpg'
        ]} />

        <motion.section 
          className="classes-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title">Ваши занятия</h2>
          {classes.length > 0 ? (
            <div className="classes-grid">
              {classes.map((classItem, index) => (
                <motion.div 
                  key={classItem.id} 
                  className="class-card"
                  initial={{ opacity: 0, scale: 0.8, rotate: -10 }}
                  animate={{ opacity: 1, scale: 1, rotate: 0 }}
                  transition={{ delay: index * 0.2, duration: 0.6, type: "spring" }}
                  whileHover={{ 
                    y: -8,
                    rotate: 3,
                    transition: { duration: 0.3 }
                  }}
                  style={{ 
                    borderLeft: '6px solid var(--primary)',
                    background: 'linear-gradient(135deg, rgba(13, 17, 23, 0.99) 0%, rgba(18, 22, 28, 0.98) 100%)'
                  }}
                >
                  <div className="class-header">
                    <div className="class-style-badge" style={{ background: 'rgba(230, 57, 70, 0.3)', color: '#E63946' }}>
                      {classItem.style}
                    </div>
                    <div className="class-rating">
                      {Array.from({ length: 5 }, (_, i) => (
                        <Star
                          key={i}
                          size={24}
                          fill={i < classItem.rating ? '#E63946' : 'none'}
                          stroke={i < classItem.rating ? '#E63946' : '#666666'}
                          strokeWidth={2.5}
                        />
                      ))}
                    </div>
                  </div>
                  <div className="class-content">
                    <div className="class-info-item">
                      <User size={28} strokeWidth={2.5} color="#F77F00" />
                      <span className="class-info-text">{classItem.instructor}</span>
                    </div>
                    <div className="class-info-item">
                      <Calendar size={28} strokeWidth={2.5} color="#FCBF49" />
                      <span className="class-info-text">
                        {new Date(classItem.date).toLocaleDateString('ru-RU', { 
                          day: 'numeric', 
                          month: 'long', 
                          year: 'numeric' 
                        })}
                      </span>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <div className="empty-icon-circle">
                <Music size={100} strokeWidth={1.5} />
              </div>
              <h3 className="empty-title">Нет занятий</h3>
              <p className="empty-text">Запишитесь на занятие прямо сейчас</p>
            </div>
          )}
        </motion.section>
      </div>
    </div>
  )
}

export default MyClasses

